<?php
session_start();
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

// Initialize database if not exists
try {
    $pdo = new PDO('sqlite:jobs.db');
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Create table if not exists
    $pdo->exec('
        CREATE TABLE IF NOT EXISTS jobs (
            job_id INTEGER PRIMARY KEY AUTOINCREMENT,
            name_jobs TEXT NOT NULL,
            text_jobs TEXT NOT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        )
    ');
} catch (PDOException $e) {
    die('Ошибка инициализации базы данных: ' . $e->getMessage());
}

$edit_id = isset($_GET['edit']) ? (int)$_GET['edit'] : null;
$edit_job = null;

// Handle messages from session
$success = $_SESSION['success'] ?? null;
$error = $_SESSION['error'] ?? null;
unset($_SESSION['success'], $_SESSION['error']);

// Handle delete
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete' && isset($_POST['job_id'])) {
    $job_id = (int)$_POST['job_id'];
    try {
        $pdo = new PDO('sqlite:jobs.db');
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $stmt = $pdo->prepare('DELETE FROM jobs WHERE job_id = :job_id');
        $stmt->execute(['job_id' => $job_id]);
        $_SESSION['success'] = 'Вакансия успешно удалена!';
        header('Location: vacancies.php');
        exit();
    } catch (PDOException $e) {
        $_SESSION['error'] = 'Ошибка удаления: ' . $e->getMessage();
        header('Location: vacancies.php');
        exit();
    }
}

// Handle edit/update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update' && isset($_POST['job_id']) && isset($_POST['name_jobs']) && isset($_POST['text_jobs'])) {
    $job_id = (int)$_POST['job_id'];
    $name_jobs = trim($_POST['name_jobs']);
    $text_jobs = trim($_POST['text_jobs']);
    
    if (!empty($name_jobs) && !empty($text_jobs)) {
        try {
            $pdo = new PDO('sqlite:jobs.db');
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $stmt = $pdo->prepare('UPDATE jobs SET name_jobs = :name_jobs, text_jobs = :text_jobs WHERE job_id = :job_id');
            $stmt->execute([
                'name_jobs' => $name_jobs,
                'text_jobs' => $text_jobs,
                'job_id' => $job_id
            ]);
            $_SESSION['success'] = 'Вакансия успешно обновлена!';
            header('Location: vacancies.php');
            exit();
        } catch (PDOException $e) {
            $error = 'Ошибка обновления: ' . $e->getMessage();
            $edit_id = $job_id; // Stay in edit mode on error
        }
    } else {
        $error = 'Заполните все поля!';
        $edit_id = $job_id; // Stay in edit mode on error
    }
}

// Handle add
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add' && isset($_POST['name_jobs']) && isset($_POST['text_jobs'])) {
    $name_jobs = trim($_POST['name_jobs']);
    $text_jobs = trim($_POST['text_jobs']);
    
    if (!empty($name_jobs) && !empty($text_jobs)) {
        try {
            $pdo = new PDO('sqlite:jobs.db');
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $stmt = $pdo->prepare('INSERT INTO jobs (name_jobs, text_jobs) VALUES (:name_jobs, :text_jobs)');
            $stmt->execute([
                'name_jobs' => $name_jobs,
                'text_jobs' => $text_jobs
            ]);
            $_SESSION['success'] = 'Вакансия успешно добавлена!';
            header('Location: vacancies.php');
            exit();
        } catch (PDOException $e) {
            $error = 'Ошибка добавления: ' . $e->getMessage();
        }
    } else {
        $error = 'Заполните все поля!';
    }
}

// Fetch existing jobs
try {
    $pdo = new PDO('sqlite:jobs.db');
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $stmt = $pdo->query('SELECT * FROM jobs ORDER BY job_id ASC');
    $jobs = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if ($edit_id) {
        $edit_stmt = $pdo->prepare('SELECT * FROM jobs WHERE job_id = :id');
        $edit_stmt->execute(['id' => $edit_id]);
        $edit_job = $edit_stmt->fetch(PDO::FETCH_ASSOC);
        if (!$edit_job) {
            $edit_id = null; // Invalid ID
        }
    }
} catch (PDOException $e) {
    $jobs = [];
    $error = 'Ошибка загрузки вакансий: ' . $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Вакансии - Админ-панель</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-blue-50 to-indigo-100 min-h-screen">
    <?php include 'header.php'; ?>
    
    <!-- Main Content -->
    <main class="max-w-7xl mx-auto py-12 px-4 sm:px-6 lg:px-8">
        <div class="bg-white rounded-2xl shadow-lg p-8">
            <h2 class="text-2xl font-bold text-gray-900 mb-6">Вакансии</h2>
            
            <?php if ($success): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <!-- Edit Form -->
            <?php if ($edit_id && $edit_job): ?>
                <div class="mb-8 p-6 bg-yellow-50 rounded-lg border border-yellow-200">
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Редактировать вакансию (ID: <?php echo $edit_job['job_id']; ?>)</h3>
                    <form method="POST" class="space-y-4">
                        <input type="hidden" name="action" value="update">
                        <input type="hidden" name="job_id" value="<?php echo $edit_job['job_id']; ?>">
                        <div>
                            <label for="name_jobs" class="block text-sm font-medium text-gray-700">Название вакансии</label>
                            <input type="text" id="name_jobs" name="name_jobs" required class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500" value="<?php echo htmlspecialchars($edit_job['name_jobs']); ?>">
                        </div>
                        <div>
                            <label for="text_jobs" class="block text-sm font-medium text-gray-700">Текст вакансии</label>
                            <textarea id="text_jobs" name="text_jobs" required rows="6" class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500"><?php echo htmlspecialchars($edit_job['text_jobs']); ?></textarea>
                        </div>
                        <div class="flex space-x-3">
                            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-medium transition duration-200">
                                Сохранить изменения
                            </button>
                            <a href="vacancies.php" class="bg-gray-500 hover:bg-gray-600 text-white px-6 py-2 rounded-lg font-medium transition duration-200 text-center">
                                Отмена
                            </a>
                        </div>
                    </form>
                </div>
            <?php endif; ?>
            
            <!-- Add Job Form -->
            <?php if (!$edit_id): ?>
                <div class="mb-8 p-6 bg-gray-50 rounded-lg">
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Добавить вакансию</h3>
                    <form method="POST" class="space-y-4">
                        <input type="hidden" name="action" value="add">
                        <div>
                            <label for="name_jobs" class="block text-sm font-medium text-gray-700">Введите название вакансии</label>
                            <input type="text" id="name_jobs" name="name_jobs" required class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500" value="<?php echo htmlspecialchars($_POST['name_jobs'] ?? ''); ?>">
                        </div>
                        <div>
                            <label for="text_jobs" class="block text-sm font-medium text-gray-700">Введите текст вакансии</label>
                            <textarea id="text_jobs" name="text_jobs" required rows="6" class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500"><?php echo htmlspecialchars($_POST['text_jobs'] ?? ''); ?></textarea>
                        </div>
                        <button type="submit" class="bg-green-600 hover:bg-green-700 text-white px-6 py-2 rounded-lg font-medium transition duration-200">
                            Сохранить
                        </button>
                    </form>
                </div>
            <?php endif; ?>
            
            <!-- Existing Jobs Table -->
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Название вакансии</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Текст вакансии</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Дата создания</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Действия</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php if (empty($jobs)): ?>
                            <tr>
                                <td colspan="5" class="px-6 py-4 text-center text-gray-500">Нет вакансий</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($jobs as $job): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($job['job_id']); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($job['name_jobs']); ?></td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <div class="max-w-xs truncate" title="<?php echo htmlspecialchars($job['text_jobs']); ?>">
                                            <?php echo htmlspecialchars($job['text_jobs']); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php echo htmlspecialchars($job['created_at'] ?? 'Не указано'); ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                                        <a href="?edit=<?php echo $job['job_id']; ?>" class="text-blue-600 hover:text-blue-900">
                                            Редактировать
                                        </a>
                                        <form method="POST" class="inline" onsubmit="return confirm('Удалить вакансию \"<?php echo htmlspecialchars($job['name_jobs']); ?>\"?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="job_id" value="<?php echo $job['job_id']; ?>">
                                            <button type="submit" class="text-red-600 hover:text-red-900">
                                                Удалить
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>
</body>
</html>