<?php
session_start();
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

// Handle status toggle and delete
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['user_id']) && isset($_POST['action'])) {
    $user_id = (int)$_POST['user_id'];
    $action = $_POST['action'];
    
    try {
        $pdo = new PDO('sqlite:data.db');
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        if ($action === 'update_balance') {
            // Handle balance update
            $balance = floatval($_POST['balance']);
            $referal_balance = floatval($_POST['referal_balance']);
            $referal_count = intval($_POST['referal_count']);
            
            $stmt = $pdo->prepare('UPDATE users SET balance = :balance, referal_balance = :referal_balance, referal = :referal WHERE user_id = :id');
            $stmt->execute([
                'balance' => $balance,
                'referal_balance' => $referal_balance,
                'referal' => $referal_count,
                'id' => $user_id
            ]);
            
            $_SESSION['success'] = 'Данные пользователя обновлены успешно!';
        } elseif ($action === 'delete') {
            // Handle delete user
            $stmt = $pdo->prepare('DELETE FROM users WHERE user_id = :id');
            $stmt->execute(['id' => $user_id]);
            
            $_SESSION['success'] = 'Пользователь успешно удален!';
        } else {
            // Handle status toggle
            $new_status = ($action === 'block') ? false : true;
            $stmt = $pdo->prepare('UPDATE users SET user_status = :status WHERE user_id = :id');
            $stmt->execute(['status' => $new_status, 'id' => $user_id]);
        }
        
        // Redirect to avoid resubmit
        header('Location: users.php');
        exit();
    } catch (PDOException $e) {
        // Simple error handling; in production, log it
        echo '<script>alert("Ошибка обновления: ' . htmlspecialchars($e->getMessage()) . '");</script>';
    }
}

// Fetch users
try {
    $pdo = new PDO('sqlite:data.db');
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $stmt = $pdo->query('SELECT * FROM users ORDER BY user_id ASC');
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $users = [];
    echo '<script>alert("Ошибка загрузки пользователей: ' . htmlspecialchars($e->getMessage()) . '");</script>';
}

$success = $_SESSION['success'] ?? null;
$error = $_SESSION['error'] ?? null;
unset($_SESSION['success'], $_SESSION['error']);
?>
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Пользователи - Админ-панель</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-blue-50 to-indigo-100 min-h-screen">
    <?php include 'header.php'; ?>
    
    <!-- Main Content -->
    <main class="max-w-7xl mx-auto py-12 px-4 sm:px-6 lg:px-8">
        <div class="bg-white rounded-2xl shadow-lg p-8">
            <h2 class="text-2xl font-bold text-gray-900 mb-6">Список пользователей</h2>
            
            <?php if ($success): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Имя</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Фамилия</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Username</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Рефералов</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Баланс рефералов</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Реф. ссылка</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Баланс</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Заказов</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Статус</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Действия</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php if (empty($users)): ?>
                            <tr>
                                <td colspan="11" class="px-6 py-4 text-center text-gray-500">Нет пользователей</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($users as $user): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($user['user_id']); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($user['first_name']); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($user['last_name']); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($user['username']); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($user['referal'] ?? 0); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($user['referal_balance'] ?? 0); ?> $</td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <a href="<?php echo htmlspecialchars($user['referal_link'] ?? '#'); ?>" target="_blank" class="text-blue-600 hover:text-blue-900 truncate max-w-xs block" title="<?php echo htmlspecialchars($user['referal_link'] ?? ''); ?>">
                                            <?php echo htmlspecialchars($user['referal_link'] ?? 'Не указано'); ?>
                                        </a>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($user['balance'] ?? 0); ?> $</td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($user['orders'] ?? 0); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full <?php echo ($user['user_status'] == true) ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                            <?php echo ($user['user_status'] == true) ? 'Активный' : 'Заблокированный'; ?>
                                        </span>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                                        <!-- Edit Button -->
                                        <button type="button" 
                                                onclick="openEditModal(<?php echo htmlspecialchars(json_encode($user)); ?>)"
                                                class="text-blue-600 hover:text-blue-900 font-medium">
                                            Редактировать
                                        </button>
                                        
                                        <!-- Block/Unblock Form -->
                                        <form method="POST" class="inline" onsubmit="return confirm('<?php echo ($user['user_status'] == true) ? 'Заблокировать пользователя?' : 'Разблокировать пользователя?'; ?>');">
                                            <input type="hidden" name="user_id" value="<?php echo $user['user_id']; ?>">
                                            <input type="hidden" name="action" value="<?php echo ($user['user_status'] == true) ? 'block' : 'unblock'; ?>">
                                            <button type="submit" class="text-<?php echo ($user['user_status'] == true) ? 'red' : 'green'; ?>-600 hover:text-<?php echo ($user['user_status'] == true) ? 'red' : 'green'; ?>-900">
                                                <?php echo ($user['user_status'] == true) ? 'Заблокировать' : 'Разблокировать'; ?>
                                            </button>
                                        </form>

                                        <!-- Delete Form -->
                                        <form method="POST" class="inline" onsubmit="return confirm('Удалить пользователя <?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>? Это действие нельзя отменить.');">
                                            <input type="hidden" name="user_id" value="<?php echo $user['user_id']; ?>">
                                            <input type="hidden" name="action" value="delete">
                                            <button type="submit" class="text-red-600 hover:text-red-900 font-medium">
                                                Удалить
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>

    <!-- Edit Modal -->
    <div id="editModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Редактирование пользователя</h3>
                
                <form id="editForm" method="POST">
                    <input type="hidden" name="user_id" id="edit_user_id">
                    <input type="hidden" name="action" value="update_balance">
                    
                    <div class="space-y-4">
                        <div>
                            <label for="edit_balance" class="block text-sm font-medium text-gray-700">Баланс ($)</label>
                            <input type="number" step="0.01" name="balance" id="edit_balance" 
                                   class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                        </div>
                        
                        <div>
                            <label for="edit_referal_balance" class="block text-sm font-medium text-gray-700">Баланс рефералов ($)</label>
                            <input type="number" step="0.01" name="referal_balance" id="edit_referal_balance" 
                                   class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                        </div>
                        
                        <div>
                            <label for="edit_referal_count" class="block text-sm font-medium text-gray-700">Количество рефералов</label>
                            <input type="number" name="referal_count" id="edit_referal_count" 
                                   class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-blue-500 focus:border-blue-500">
                        </div>
                    </div>
                    
                    <div class="flex justify-end space-x-3 mt-6">
                        <button type="button" 
                                onclick="closeEditModal()" 
                                class="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-200 hover:bg-gray-300 rounded-md transition duration-200">
                            Отмена
                        </button>
                        <button type="submit" 
                                class="px-4 py-2 text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 rounded-md transition duration-200">
                            Сохранить
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        function openEditModal(user) {
            document.getElementById('edit_user_id').value = user.user_id;
            document.getElementById('edit_balance').value = user.balance || 0;
            document.getElementById('edit_referal_balance').value = user.referal_balance || 0;
            document.getElementById('edit_referal_count').value = user.referal || 0;
            document.getElementById('editModal').classList.remove('hidden');
        }

        function closeEditModal() {
            document.getElementById('editModal').classList.add('hidden');
        }

        // Close modal when clicking outside
        document.getElementById('editModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeEditModal();
            }
        });

        // Close modal with Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeEditModal();
            }
        });
    </script>
</body>
</html>