<?php
session_start();
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

// Handle messages from session
$success = $_SESSION['success'] ?? null;
$error = $_SESSION['error'] ?? null;
unset($_SESSION['success'], $_SESSION['error']);

// Fetch orders with product names and user info
try {
    $pdo = new PDO('sqlite:data.db');
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    $stmt = $pdo->query('
        SELECT 
            o.*,
            p.product_name,
            u.first_name,
            u.last_name,
            u.username
        FROM orders o
        LEFT JOIN products p ON o.product_id = p.product_id
        LEFT JOIN users u ON o.order_user_id = u.user_id
        ORDER BY o.order_id DESC
    ');
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $orders = [];
    $error = 'Ошибка загрузки заказов: ' . $e->getMessage();
}

// Function to get status text
function getStatusText($status) {
    switch ($status) {
        case 'load':
            return '<span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-yellow-100 text-yellow-800">В ожидании</span>';
        case 'completed':
            return '<span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">Завершен</span>';
        case 'cancelled':
            return '<span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-red-100 text-red-800">Отменен</span>';
        default:
            return '<span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-gray-100 text-gray-800">' . htmlspecialchars($status) . '</span>';
    }
}

// Function to get payment method text
function getPaymentMethodText($method) {
    switch ($method) {
        case 'card':
            return 'Банковская карта';
        case 'cryptobot':
            return 'CryptoBot';
        case 'balance':
            return 'С баланса';
        default:
            return htmlspecialchars($method);
    }
}
?>
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Заказы - Админ-панель</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-blue-50 to-indigo-100 min-h-screen">
    <?php include 'header.php'; ?>
    
    <!-- Main Content -->
    <main class="max-w-7xl mx-auto py-12 px-4 sm:px-6 lg:px-8">
        <div class="bg-white rounded-2xl shadow-lg p-8">
            <h2 class="text-2xl font-bold text-gray-900 mb-6">Список заказов</h2>
            
            <?php if ($success): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID Заказа</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Товар</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Цена</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Способ оплаты</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Адрес доставки</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Статус</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Пользователь</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Дата и время</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Тип доставки</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php if (empty($orders)): ?>
                            <tr>
                                <td colspan="9" class="px-6 py-4 text-center text-gray-500">Нет заказов</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($orders as $order): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        #<?php echo htmlspecialchars($order['order_id']); ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <div class="font-medium"><?php echo htmlspecialchars($order['product_name'] ?? 'Товар не найден'); ?></div>
                                        <div class="text-gray-500 text-xs">ID: <?php echo htmlspecialchars($order['product_id']); ?></div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php echo htmlspecialchars($order['price_order']); ?> $
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php echo getPaymentMethodText($order['payment_method']); ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <?php if (!empty($order['delivery_adress'])): ?>
                                            <span class="bg-blue-50 text-blue-700 px-2 py-1 rounded text-xs">
                                                <?php echo htmlspecialchars($order['delivery_adress']); ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="text-gray-400 text-xs">Не указан</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <?php echo getStatusText($order['order_status']); ?>
                                    </td>
                                    <td class="px-6 py-4 text-sm text-gray-900">
                                        <div class="font-medium">
                                            <?php echo htmlspecialchars($order['first_name'] ?? 'Неизвестно'); ?>
                                            <?php echo htmlspecialchars($order['last_name'] ?? ''); ?>
                                        </div>
                                        <div class="text-gray-500 text-xs">
                                            @<?php echo htmlspecialchars($order['username'] ?? 'N/A'); ?>
                                        </div>
                                        <div class="text-gray-400 text-xs">
                                            ID: <?php echo htmlspecialchars($order['order_user_id']); ?>
                                        </div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <div><?php echo htmlspecialchars($order['order_date'] ?? 'Не указана'); ?></div>
                                        <div class="text-gray-500 text-xs"><?php echo htmlspecialchars($order['order_time'] ?? ''); ?></div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                        <?php if (!empty($order['tip_delivery'])): ?>
                                            <span class="bg-green-50 text-green-700 px-2 py-1 rounded text-xs">
                                                <?php echo htmlspecialchars($order['tip_delivery']); ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="text-gray-400 text-xs">Не указан</span>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            
            <!-- Statistics -->
            <?php if (!empty($orders)): ?>
                <div class="mt-8 grid grid-cols-1 md:grid-cols-4 gap-4">
                    <div class="bg-blue-50 p-4 rounded-lg">
                        <div class="text-sm font-medium text-blue-800">Всего заказов</div>
                        <div class="text-2xl font-bold text-blue-900"><?php echo count($orders); ?></div>
                    </div>
                    <div class="bg-yellow-50 p-4 rounded-lg">
                        <div class="text-sm font-medium text-yellow-800">В ожидании</div>
                        <div class="text-2xl font-bold text-yellow-900">
                            <?php echo count(array_filter($orders, function($order) { return $order['order_status'] === 'load'; })); ?>
                        </div>
                    </div>
                    <div class="bg-green-50 p-4 rounded-lg">
                        <div class="text-sm font-medium text-green-800">Завершено</div>
                        <div class="text-2xl font-bold text-green-900">
                            <?php echo count(array_filter($orders, function($order) { return $order['order_status'] === 'completed'; })); ?>
                        </div>
                    </div>
                    <div class="bg-red-50 p-4 rounded-lg">
                        <div class="text-sm font-medium text-red-800">Отменено</div>
                        <div class="text-2xl font-bold text-red-900">
                            <?php echo count(array_filter($orders, function($order) { return $order['order_status'] === 'cancelled'; })); ?>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </main>
</body>
</html>