import logging
import sqlite3
from aiogram import Router, F
from aiogram.types import CallbackQuery, Message, InlineKeyboardMarkup, InlineKeyboardButton
from db import get_user_language, get_language_text, get_button_text, DATA_DB

logger = logging.getLogger(__name__)

router = Router(name="ref_sistema")

def build_back_keyboard(lang: str) -> InlineKeyboardMarkup:
    """
    Создает клавиатуру с кнопкой button9 (назад)
    """
    button9_text = get_button_text(lang, "button9", "⬅️ Назад")
    
    keyboard = InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text=button9_text, callback_data="menu:button9")]
        ]
    )
    
    return keyboard

def get_user_ref_data(user_id: int) -> dict:
    """
    Получает реферальные данные пользователя из базы данных
    """
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute(
            "SELECT referal_link, referal, referal_balance FROM users WHERE user_id = ?", 
            (user_id,)
        )
        row = cur.fetchone()
        conn.close()
        
        if row:
            return {
                'referal_link': row[0] or "",
                'referal': row[1] or 0,
                'referal_balance': row[2] or 0.0
            }
        else:
            return {
                'referal_link': "",
                'referal': 0,
                'referal_balance': 0.0
            }
    except Exception as e:
        logger.error(f"Error getting user ref data for {user_id}: {e}")
        return {
            'referal_link': "",
            'referal': 0,
            'referal_balance': 0.0
        }

@router.callback_query(F.data == "menu:button7")
async def show_ref_sistema(cb: CallbackQuery):
    """
    Показывает реферальную систему при нажатии на button7
    """
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    print(f"=== REF SISTEMA ===")
    print(f"User: {user_id}")
    
    # Получаем реферальные данные пользователя
    ref_data = get_user_ref_data(user_id)
    
    print(f"Ref data: {ref_data}")
    
    # Получаем текст text20 из базы данных
    text20_template = get_language_text(lang, "text20") or "Реферальная система\nРеферальная ссылка: {referal_link}\nКоличество рефералов: {referal}\nРеферальный баланс: {referal_balance}$"
    
    # Заменяем плейсхолдеры на реальные значения
    text20 = text20_template.format(
        referal_link=ref_data['referal_link'],
        referal=ref_data['referal'],
        referal_balance=ref_data['referal_balance']
    )
    
    # Создаем клавиатуру с кнопкой назад
    keyboard = build_back_keyboard(lang)
    
    # Удаляем предыдущее сообщение
    try:
        await cb.message.delete()
    except Exception:
        pass
    
    # Отправляем сообщение с реферальной системой
    await cb.message.answer(text20, reply_markup=keyboard)
    
    await cb.answer()