import logging
import sqlite3
import json
import os
from datetime import datetime

logger = logging.getLogger(__name__)

def load_chat_id():
    """
    Загружает chat_id из файла chat_id.json
    """
    try:
        current_dir = os.path.dirname(os.path.abspath(__file__))
        config_path = os.path.join(current_dir, 'chat_id.json')
        
        if os.path.exists(config_path):
            with open(config_path, 'r', encoding='utf-8') as f:
                config = json.load(f)
                return config.get('chat_id')
        else:
            print("chat_id.json not found")
            return None
    except Exception as e:
        logger.error(f"Error loading chat_id.json: {e}")
        return None

async def send_order_notification(bot, order_data: dict):
    """
    Отправляет уведомление о новом заказе в указанный чат
    """
    try:
        chat_id = load_chat_id()
        if not chat_id:
            print("No chat_id found for notifications")
            return
        
        message_text = "📦 Поступил новый заказ\nВойдите в раздел заказы в админ панели."
        
        await bot.send_message(chat_id=chat_id, text=message_text)
        print(f"Order notification sent to chat_id: {chat_id}")
        
    except Exception as e:
        logger.error(f"Error sending order notification: {e}")

def update_user_orders_count(user_id: int):
    """
    Увеличивает счетчик заказов пользователя на +1
    """
    try:
        conn = sqlite3.connect('data.db')
        cur = conn.cursor()
        cur.execute("UPDATE users SET orders = orders + 1 WHERE user_id = ?", (user_id,))
        conn.commit()
        conn.close()
        print(f"User {user_id} orders count increased by 1")
        return True
    except Exception as e:
        logger.error(f"Error updating user orders count: {e}")
        return False

def create_orders_table():
    """
    Создает таблицу orders если её нет
    """
    try:
        conn = sqlite3.connect('data.db')
        cur = conn.cursor()
        
        # Проверяем существование колонки tip_delivery
        cur.execute("PRAGMA table_info(orders)")
        columns = [column[1] for column in cur.fetchall()]
        
        cur.execute('''
            CREATE TABLE IF NOT EXISTS orders (
                order_id INTEGER PRIMARY KEY AUTOINCREMENT,
                product_id INTEGER,
                price_order REAL,
                payment_method TEXT,
                delivery_adress TEXT,
                order_status TEXT,
                order_user_id INTEGER,
                order_manager_id TEXT,
                order_date TEXT,
                order_time TEXT,
                tip_delivery TEXT
            )
        ''')
        
        # Добавляем колонку tip_delivery если её нет
        if 'tip_delivery' not in columns:
            cur.execute('ALTER TABLE orders ADD COLUMN tip_delivery TEXT')
            print("Added tip_delivery column to orders table")
        
        conn.commit()
        conn.close()
        print("Orders table created/verified")
    except Exception as e:
        logger.error(f"Error creating orders table: {e}")

# Создаем таблицу при импорте
create_orders_table()

async def save_order(state_data: dict, user_id: int, payment_method: str = "cryptobot", bot=None):
    """
    Универсальная функция для сохранения заказа в базу данных
    
    Args:
        state_data: Данные из состояния FSM
        user_id: ID пользователя
        payment_method: Метод оплаты (cryptobot, button14, button16 и т.д.)
        bot: Экземпляр бота для отправки уведомлений
    
    Returns:
        bool: True если успешно, False если ошибка
    """
    try:
        # Логируем все данные из состояния для отладки
        print(f"=== DEBUG STATE DATA ===")
        for key, value in state_data.items():
            print(f"{key}: {value}")
        print(f"========================")
        
        conn = sqlite3.connect('data.db')
        cur = conn.cursor()
        
        # Получаем данные из состояния
        selected_product = state_data.get('selected_product')
        selected_tag = state_data.get('selected_tag')
        delivery_type = state_data.get('delivery_type')  # button11 или button12
        delivery_address = state_data.get('delivery_address', '')
        
        print(f"delivery_type from state: {delivery_type}")
        print(f"payment_method to save: {payment_method}")
        
        if not selected_product or not selected_tag:
            print("No product data in state for saving order")
            return False
        
        product_id = selected_product.get('product_id')
        price = selected_tag.get('price', 0)
        product_name = selected_product.get('product_name', 'Unknown')
        
        # Определяем текст способа доставки
        tip_delivery = ""
        if delivery_type == "button11":
            tip_delivery = "Доставка"
        elif delivery_type == "button12":
            tip_delivery = "Самовывоз"
        else:
            tip_delivery = f"Неизвестно ({delivery_type})"  # на случай если delivery_type не распознан
        
        print(f"tip_delivery to save: {tip_delivery}")
        
        # Текущая дата и время
        now = datetime.now()
        order_date = now.strftime("%Y-%m-%d")
        order_time = now.strftime("%H:%M:%S")
        
        # Сохраняем заказ в базу
        cur.execute('''
            INSERT INTO orders (
                product_id, price_order, payment_method, delivery_adress, 
                order_status, order_user_id, order_manager_id, order_date, order_time, tip_delivery
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ''', (
            product_id,
            price,
            payment_method,  # сохраняем переданный метод оплаты (button14, button16, cryptobot и т.д.)
            delivery_address,  # адрес доставки если есть
            'load',  # order_status
            user_id,
            '',  # order_manager_id - оставляем пустым
            order_date,
            order_time,
            tip_delivery  # способ доставки
        ))
        
        conn.commit()
        conn.close()
        
        print(f"Order saved to DB: product_id={product_id}, product_name={product_name}, price={price}, user_id={user_id}, payment_method={payment_method}, delivery_type={tip_delivery}")
        
        # Увеличиваем счетчик заказов пользователя
        orders_updated = update_user_orders_count(user_id)
        if orders_updated:
            print(f"Successfully updated orders count for user {user_id}")
        else:
            print(f"Failed to update orders count for user {user_id}")
        
        # Отправляем уведомление о новом заказе (для всех методов оплаты)
        if bot:
            order_data = {
                'user_id': user_id,
                'product_name': product_name,
                'price': price,
                'payment_method': payment_method,
                'delivery_type': tip_delivery
            }
            await send_order_notification(bot, order_data)
        
        return True
        
    except Exception as e:
        logger.error(f"Error saving order to DB: {e}")
        return False

async def get_user_orders(user_id: int):
    """
    Получает все заказы пользователя
    
    Args:
        user_id: ID пользователя
    
    Returns:
        list: Список заказов пользователя
    """
    try:
        conn = sqlite3.connect('data.db')
        cur = conn.cursor()
        
        cur.execute('''
            SELECT * FROM orders 
            WHERE order_user_id = ? 
            ORDER BY order_date DESC, order_time DESC
        ''', (user_id,))
        
        orders = cur.fetchall()
        conn.close()
        
        return orders
        
    except Exception as e:
        logger.error(f"Error getting user orders: {e}")
        return []

async def get_order_by_id(order_id: int):
    """
    Получает заказ по ID
    
    Args:
        order_id: ID заказа
    
    Returns:
        dict: Данные заказа или None
    """
    try:
        conn = sqlite3.connect('data.db')
        cur = conn.cursor()
        
        cur.execute('SELECT * FROM orders WHERE order_id = ?', (order_id,))
        order = cur.fetchone()
        conn.close()
        
        if order:
            return {
                'order_id': order[0],
                'product_id': order[1],
                'price_order': order[2],
                'payment_method': order[3],
                'delivery_adress': order[4],
                'order_status': order[5],
                'order_user_id': order[6],
                'order_manager_id': order[7],
                'order_date': order[8],
                'order_time': order[9],
                'tip_delivery': order[10] if len(order) > 10 else ''
            }
        return None
        
    except Exception as e:
        logger.error(f"Error getting order by ID: {e}")
        return None

async def update_order_status(order_id: int, new_status: str):
    """
    Обновляет статус заказа
    
    Args:
        order_id: ID заказа
        new_status: Новый статус
    
    Returns:
        bool: True если успешно, False если ошибка
    """
    try:
        conn = sqlite3.connect('data.db')
        cur = conn.cursor()
        
        cur.execute('''
            UPDATE orders SET order_status = ? WHERE order_id = ?
        ''', (new_status, order_id))
        
        conn.commit()
        conn.close()
        
        print(f"Order {order_id} status updated to {new_status}")
        return True
        
    except Exception as e:
        logger.error(f"Error updating order status: {e}")
        return False