import logging
import sqlite3
from aiogram import Router, F
from aiogram.types import CallbackQuery, Message, InlineKeyboardMarkup, InlineKeyboardButton
from db import get_user_language, get_language_text, get_button_text

logger = logging.getLogger(__name__)

router = Router(name="my_order")

# Callback данные
BACK_CB = "menu:button9"

def get_button9_text(lang: str) -> str:
    """Получает текст кнопки назад"""
    label = "⬅️ Назад"
    try:
        from db import MENU_DB
        conn = sqlite3.connect(MENU_DB)
        cur = conn.cursor()
        cur.execute("SELECT button9 FROM language WHERE name = ?", (lang,))
        row = cur.fetchone()
        conn.close()
        if row and row[0] and str(row[0]).strip():
            label = str(row[0]).strip()
    except Exception as e:
        logger.error(f"get_button9_text error for lang={lang}: {e}")
    return label

def build_back_keyboard(lang: str) -> InlineKeyboardMarkup:
    """Создает клавиатуру с кнопкой назад"""
    return InlineKeyboardMarkup(
        inline_keyboard=[[InlineKeyboardButton(text=get_button9_text(lang), callback_data=BACK_CB)]]
    )

def get_product_name(product_id: int) -> str:
    """Получает название товара по product_id"""
    try:
        conn = sqlite3.connect('data.db')
        cur = conn.cursor()
        cur.execute("SELECT product_name FROM products WHERE product_id = ?", (product_id,))
        row = cur.fetchone()
        conn.close()
        if row and row[0]:
            return str(row[0])
        return f"Товар #{product_id}"
    except Exception as e:
        logger.error(f"Error getting product name for id {product_id}: {e}")
        return f"Товар #{product_id}"

def get_user_orders(user_id: int) -> list:
    """Получает все заказы пользователя"""
    try:
        conn = sqlite3.connect('data.db')
        cur = conn.cursor()
        cur.execute('''
            SELECT product_id, order_date, order_time 
            FROM orders 
            WHERE order_user_id = ? 
            ORDER BY order_date DESC, order_time DESC
        ''', (user_id,))
        orders = cur.fetchall()
        conn.close()
        return orders
    except Exception as e:
        logger.error(f"Error getting user orders for {user_id}: {e}")
        return []

@router.callback_query(F.data == "menu:button3")
async def open_my_orders(cb: CallbackQuery):
    """
    Открывает мои заказы (button3)
    """
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    try:
        await cb.message.delete()
    except Exception:
        pass
    
    # Получаем заказы пользователя
    orders = get_user_orders(user_id)
    
    if not orders:
        # Если нет заказов, показываем text25
        text25 = get_language_text(lang, "text25") or "У вас пока нет заказов"
        await cb.message.answer(text25, reply_markup=build_back_keyboard(lang))
        await cb.answer()
        return
    
    # Получаем текст заголовка из text30
    text30 = get_language_text(lang, "text30") or "Ваши заказы:"
    
    # Формируем сообщение с заказами
    message_lines = [text30, ""]
    
    for order in orders:
        product_id = order[0]
        order_date = order[1]
        order_time = order[2]
        
        # Получаем название товара
        product_name = get_product_name(product_id)
        
        # Добавляем информацию о заказе
        message_lines.append(f"📦 {product_name}")
        message_lines.append(f"📅 Дата: {order_date}")
        message_lines.append(f"⏰ Время: {order_time}")
        message_lines.append("---------------")
    
    # Убираем последний разделитель
    if message_lines[-1] == "---------------":
        message_lines.pop()
    
    # Объединяем все строки в одно сообщение
    orders_message = "\n".join(message_lines)
    
    # Отправляем сообщение с заказами и кнопкой назад
    await cb.message.answer(orders_message, reply_markup=build_back_keyboard(lang))
    await cb.answer()