import logging
import sqlite3
from aiogram import Router, F
from aiogram.types import CallbackQuery, Message, InlineKeyboardMarkup, InlineKeyboardButton
from db import get_user_language, get_language_text, get_button_text

logger = logging.getLogger(__name__)

router = Router(name="jobs")

# Callback данные
BACK_CB = "menu:button9"
JOBS_CB_PREFIX = "job:"
JOBS_BACK_CB = "jobs_back"

def get_button9_text(lang: str) -> str:
    """Получает текст кнопки назад"""
    label = "⬅️ Назад"
    try:
        from db import MENU_DB
        conn = sqlite3.connect(MENU_DB)
        cur = conn.cursor()
        cur.execute("SELECT button9 FROM language WHERE name = ?", (lang,))
        row = cur.fetchone()
        conn.close()
        if row and row[0] and str(row[0]).strip():
            label = str(row[0]).strip()
    except Exception as e:
        logger.error(f"get_button9_text error for lang={lang}: {e}")
    return label

def build_back_keyboard(lang: str) -> InlineKeyboardMarkup:
    """Создает клавиатуру с кнопкой назад"""
    return InlineKeyboardMarkup(
        inline_keyboard=[[InlineKeyboardButton(text=get_button9_text(lang), callback_data=BACK_CB)]]
    )

def build_jobs_back_keyboard(lang: str) -> InlineKeyboardMarkup:
    """Создает клавиатуру для возврата к списку вакансий"""
    return InlineKeyboardMarkup(
        inline_keyboard=[[InlineKeyboardButton(text=get_button9_text(lang), callback_data=JOBS_BACK_CB)]]
    )

def get_all_jobs() -> list:
    """Получает все вакансии из jobs.db"""
    try:
        conn = sqlite3.connect('jobs.db')
        cur = conn.cursor()
        # Получаем только name_jobs, так как id нет
        cur.execute("SELECT name_jobs FROM jobs")
        jobs = cur.fetchall()
        conn.close()
        return jobs
    except Exception as e:
        logger.error(f"Error getting jobs from jobs.db: {e}")
        return []

def get_job_details(job_name: str) -> dict:
    """Получает детали вакансии по названию"""
    try:
        conn = sqlite3.connect('jobs.db')
        cur = conn.cursor()
        cur.execute("SELECT name_jobs, text_jobs FROM jobs WHERE name_jobs = ?", (job_name,))
        row = cur.fetchone()
        conn.close()
        if row:
            return {
                'name': row[0],
                'description': row[1]
            }
        return None
    except Exception as e:
        logger.error(f"Error getting job details for {job_name}: {e}")
        return None

def build_jobs_keyboard(lang: str, jobs: list) -> InlineKeyboardMarkup:
    """Создает клавиатуру со списком вакансий"""
    keyboard_rows = []
    
    for job in jobs:
        job_name = job[0]  # name_jobs находится в первом элементе кортежа
        keyboard_rows.append([
            InlineKeyboardButton(text=job_name, callback_data=f"{JOBS_CB_PREFIX}{job_name}")
        ])
    
    # Добавляем кнопку назад в конец
    keyboard_rows.append([
        InlineKeyboardButton(text=get_button9_text(lang), callback_data=BACK_CB)
    ])
    
    return InlineKeyboardMarkup(inline_keyboard=keyboard_rows)

@router.callback_query(F.data == "menu:button5")
async def open_jobs(cb: CallbackQuery):
    """
    Открывает список вакансий (button5)
    """
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    try:
        await cb.message.delete()
    except Exception:
        pass
    
    # Получаем все вакансии
    jobs = get_all_jobs()
    
    if not jobs:
        # Если нет вакансий
        no_jobs_text = "На данный момент вакансий нет"
        await cb.message.answer(no_jobs_text, reply_markup=build_back_keyboard(lang))
        await cb.answer()
        return
    
    # Получаем текст заголовка из text31
    text31 = get_language_text(lang, "text31") or "Доступные вакансии:"
    
    # Создаем клавиатуру с вакансиями
    keyboard = build_jobs_keyboard(lang, jobs)
    
    # Отправляем сообщение с вакансиями
    await cb.message.answer(text31, reply_markup=keyboard)
    await cb.answer()

@router.callback_query(F.data == JOBS_BACK_CB)
async def back_to_jobs_list(cb: CallbackQuery):
    """
    Возвращает к списку вакансий
    """
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    try:
        await cb.message.delete()
    except Exception:
        pass
    
    # Получаем все вакансии
    jobs = get_all_jobs()
    
    if not jobs:
        # Если нет вакансий
        no_jobs_text = "На данный момент вакансий нет"
        await cb.message.answer(no_jobs_text, reply_markup=build_back_keyboard(lang))
        await cb.answer()
        return
    
    # Получаем текст заголовка из text31
    text31 = get_language_text(lang, "text31") or "Доступные вакансии:"
    
    # Создаем клавиатуру с вакансиями
    keyboard = build_jobs_keyboard(lang, jobs)
    
    # Отправляем сообщение с вакансиями
    await cb.message.answer(text31, reply_markup=keyboard)
    await cb.answer()

@router.callback_query(F.data.startswith(JOBS_CB_PREFIX))
async def show_job_details(cb: CallbackQuery):
    """
    Показывает детали вакансии
    """
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    # Получаем название вакансии из callback_data
    job_name = cb.data[len(JOBS_CB_PREFIX):]
    
    # Получаем детали вакансии
    job = get_job_details(job_name)
    
    if not job:
        await cb.answer("Вакансия не найдена", show_alert=True)
        return
    
    try:
        await cb.message.delete()
    except Exception:
        pass
    
    # Формируем сообщение с деталями вакансии
    message_text = f"💼 {job['name']}\n\n{job['description']}"
    
    # Создаем клавиатуру для возврата к списку вакансий
    keyboard = build_jobs_back_keyboard(lang)
    
    # Отправляем сообщение с деталями вакансии
    await cb.message.answer(message_text, reply_markup=keyboard)
    await cb.answer()