import logging
import os
import sqlite3
from aiogram import Router, F
from aiogram.types import (
    CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton, Message
)
from aiogram.types.input_file import FSInputFile
from aiogram.fsm.state import StatesGroup, State
from aiogram.fsm.context import FSMContext
from db import get_user_language, get_language_text, MENU_DB, DATA_DB

router = Router(name="catalog")
logger = logging.getLogger(__name__)

BACK_CB = "menu:button9"
CITY_CB_PREFIX = "city:"
PROD_CB_PREFIX = "prod:"
TAG_CB_PREFIX = "tag:"
DELIVERY_CB_PREFIX = "delivery:"
PAYMENT_CB_PREFIX = "payment:"

class CatalogStates(StatesGroup):
    wait_city = State()

def get_button9_text(lang: str) -> str:
    label = "⬅️ Назад"
    try:
        conn = sqlite3.connect(MENU_DB)
        cur = conn.cursor()
        cur.execute("SELECT button9 FROM language WHERE name = ?", (lang,))
        row = cur.fetchone()
        conn.close()
        if row and row[0] and str(row[0]).strip():
            label = str(row[0]).strip()
    except Exception as e:
        logger.error(f"get_button9_text error for lang={lang}: {e}")
    return label

def get_button_text(lang: str, column: str, default_value: str = "") -> str:
    txt = default_value
    try:
        conn = sqlite3.connect(MENU_DB)
        cur = conn.cursor()
        cur.execute(f"SELECT {column} FROM language WHERE name = ?", (lang,))
        row = cur.fetchone()
        conn.close()
        if row and row[0] and str(row[0]).strip():
            txt = str(row[0]).strip()
    except Exception as e:
        logger.error(f"get_button_text error col={column} lang={lang}: {e}")
    return txt

def build_back_keyboard(lang: str) -> InlineKeyboardMarkup:
    return InlineKeyboardMarkup(
        inline_keyboard=[[InlineKeyboardButton(text=get_button9_text(lang), callback_data=BACK_CB)]]
    )

def load_all_cities() -> list[tuple[int, str]]:
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("SELECT city_id, city_name FROM city")
        rows = cur.fetchall()
        conn.close()
        return [(int(r[0]), str(r[1]).strip()) for r in rows if r and r[0] is not None and r[1]]
    except Exception as e:
        logger.error(f"load_all_cities error: {e}")
        return []

def get_city_name(city_id: int) -> str | None:
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("SELECT city_name FROM city WHERE city_id = ?", (city_id,))
        row = cur.fetchone()
        conn.close()
        return str(row[0]).strip() if row and row[0] else None
    except Exception as e:
        logger.error(f"get_city_name error: {e}")
        return None

def get_user_balance(user_id: int) -> float:
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("SELECT balance FROM users WHERE user_id = ?", (user_id,))
        row = cur.fetchone()
        conn.close()
        if row and row[0] is not None:
            try:
                return float(row[0])
            except Exception:
                return 0.0
    except Exception as e:
        logger.error(f"get_user_balance error user_id={user_id}: {e}")
        return 0.0

def update_user_balance(user_id: int, amount: float) -> bool:
    """
    Обновляет баланс пользователя
    """
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("UPDATE users SET balance = balance - ? WHERE user_id = ?", (amount, user_id))
        conn.commit()
        conn.close()
        return True
    except Exception as e:
        logger.error(f"update_user_balance error user_id={user_id}: {e}")
        return False

def find_cities_by_prefix(prefix: str) -> list[tuple[int, str]]:
    q = (prefix or "").strip().lower()
    if not q:
        return []
    cities = load_all_cities()
    matches = [(cid, name) for cid, name in cities if name.lower().startswith(q)]
    return sorted(matches, key=lambda x: x[1].lower())

def find_products_for_city(city_id: int) -> list[tuple[int, str]]:
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("""
        SELECT product_id, product_name
        FROM products
        WHERE (',' || REPLACE(product_city, ' ', '') || ',') LIKE '%,' || ? || ',%'
        ORDER BY product_name COLLATE NOCASE
        """, (str(city_id),))
        rows = cur.fetchall()
        conn.close()
        return [(int(r[0]), str(r[1]).strip()) for r in rows if r and r[0] is not None and r[1]]
    except Exception as e:
        logger.error(f"find_products_for_city error city_id={city_id}: {e}")
        return []

def get_product(product_id: int):
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("""
        SELECT product_id, product_name, product_image, product_description, product_tags, product_tags2, product_tags3, product_tags4
        FROM products
        WHERE product_id = ?
        """, (product_id,))
        row = cur.fetchone()
        conn.close()
        if not row:
            return None
        return {
            "product_id": int(row[0]),
            "product_name": (row[1] or "").strip(),
            "product_image": (row[2] or "").strip() if row[2] else "",
            "product_description": (row[3] or "").strip() if row[3] else "",
            "product_tags": (row[4] or "").strip() if row[4] else "",
            "product_tags2": (row[5] or "").strip() if row[5] else "",
            "product_tags3": (row[6] or "").strip() if row[6] else "",
            "product_tags4": (row[7] or "").strip() if row[7] else "",
        }
    except Exception as e:
        logger.error(f"get_product error id={product_id}: {e}")
        return None

def build_cities_keyboard(lang: str, cities: list[tuple[int, str]]) -> InlineKeyboardMarkup:
    rows: list[list[InlineKeyboardButton]] = []
    for cid, name in cities:
        rows.append([InlineKeyboardButton(text=name, callback_data=f"{CITY_CB_PREFIX}{cid}")])
    rows.append([InlineKeyboardButton(text=get_button9_text(lang), callback_data=BACK_CB)])
    return InlineKeyboardMarkup(inline_keyboard=rows)

def build_products_keyboard(lang: str, products: list[tuple[int, str]], city_id: int) -> InlineKeyboardMarkup:
    rows: list[list[InlineKeyboardButton]] = []
    for pid, name in products:
        rows.append([InlineKeyboardButton(text=name, callback_data=f"{PROD_CB_PREFIX}{pid}:{city_id}")])
    rows.append([InlineKeyboardButton(text=get_button9_text(lang), callback_data=BACK_CB)])
    return InlineKeyboardMarkup(inline_keyboard=rows)

def parse_tag_value(raw: str) -> tuple[str, str] | None:
    if not raw:
        return None
    parts = [p.strip() for p in raw.split(",")]
    if not parts or not parts[0]:
        return None
    name = parts[0]
    price = parts[1] if len(parts) > 1 and parts[1] else ""
    if price:
        return (name, f"{price} $")
    return (name, "")

def parse_tag_price_numeric(raw: str) -> float | None:
    if not raw:
        return None
    parts = [p.strip() for p in raw.split(",")]
    if len(parts) < 2 or not parts[1]:
        return None
    pr = parts[1].replace(",", ".")
    try:
        return float(pr)
    except Exception:
        return None

def build_product_options_keyboard(lang: str, product: dict) -> InlineKeyboardMarkup:
    rows: list[list[InlineKeyboardButton]] = []
    tags_raw = [
        product.get("product_tags", ""),
        product.get("product_tags2", ""),
        product.get("product_tags3", ""),
        product.get("product_tags4", ""),
    ]
    idx = 0
    for raw in tags_raw:
        parsed = parse_tag_value(raw)
        if not parsed:
            idx += 1
            continue
        label = parsed[0]
        price = parsed[1]
        text = f"{label} — {price}" if price else label
        rows.append([InlineKeyboardButton(text=text, callback_data=f"{TAG_CB_PREFIX}{product['product_id']}:{idx}")])
        idx += 1
    rows.append([InlineKeyboardButton(text=get_button9_text(lang), callback_data=BACK_CB)])
    return InlineKeyboardMarkup(inline_keyboard=rows)

def build_delivery_keyboard(lang: str) -> InlineKeyboardMarkup:
    b11 = get_button_text(lang, "button11", "🚗 Доставка")
    b12 = get_button_text(lang, "button12", "🏃 Самовывоз")
    b9 = get_button9_text(lang)
    
    return InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text=b11, callback_data="delivery:delivery")],
            [InlineKeyboardButton(text=b12, callback_data="delivery:pickup")],
            [InlineKeyboardButton(text=b9, callback_data=BACK_CB)]
        ]
    )

def build_payment_keyboard(lang: str, show_balance: bool = False) -> InlineKeyboardMarkup:
    b13 = get_button_text(lang, "button13", "💳 CryptoBot")
    b14 = get_button_text(lang, "button14", "💳 Банковская карта")
    b9 = get_button9_text(lang)
    
    rows = [
        [InlineKeyboardButton(text=b13, callback_data="payment:cryptobot")],
        [InlineKeyboardButton(text=b14, callback_data="payment:card")],
    ]
    
    if show_balance:
        b16 = get_button_text(lang, "button16", "💰 Оплатить с баланса")
        rows.append([InlineKeyboardButton(text=b16, callback_data="payment:balance")])
    
    rows.append([InlineKeyboardButton(text=b9, callback_data=BACK_CB)])
    
    return InlineKeyboardMarkup(inline_keyboard=rows)

def resolve_image_path(raw_path: str) -> tuple[bool, str | None]:
    if not raw_path:
        return False, None
    p = raw_path.strip()
    if p.lower().startswith(("http://", "https://")):
        return True, p
    candidates = []
    if p.startswith("../"):
        candidates.append(p)
    else:
        if p.startswith("upload/"):
            candidates.append("../" + p)
        candidates.append(p)
        candidates.append("./" + p)
    for c in candidates:
        norm = os.path.normpath(c)
        if os.path.exists(norm):
            return False, norm
    return False, os.path.normpath(candidates[0])

@router.callback_query(F.data == "menu:button1")
async def open_catalog(cb: CallbackQuery, state: FSMContext):
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    text2 = get_language_text(lang, "text2") or "Введите название города"
    try:
        await cb.message.delete()
    except Exception:
        pass
    prompt_msg = await cb.message.answer(text2, reply_markup=build_back_keyboard(lang))
    await state.set_state(CatalogStates.wait_city)
    await state.update_data(prompt_msg_id=prompt_msg.message_id)
    await cb.answer()

@router.message(CatalogStates.wait_city)
async def handle_city_input(message: Message, state: FSMContext):
    user_id = message.from_user.id
    lang = get_user_language(user_id)
    user_text = (message.text or "").strip()
    try:
        await message.delete()
    except Exception:
        pass

    try:
        data = await state.get_data()
        prompt_id = data.get("prompt_msg_id")
        if prompt_id:
            await message.bot.delete_message(chat_id=message.chat.id, message_id=prompt_id)
    except Exception:
        pass

    matches = find_cities_by_prefix(user_text)
    if matches:
        text4 = get_language_text(lang, "text4") or "Найдены города"
        kb = build_cities_keyboard(lang, matches)
        await message.answer(text4, reply_markup=kb)
        await state.clear()
        return
    text3 = get_language_text(lang, "text3") or "Город не найден. Попробуйте снова."
    await message.answer(text3, reply_markup=build_back_keyboard(lang))

@router.callback_query(F.data.startswith(CITY_CB_PREFIX))
async def on_city_clicked(cb: CallbackQuery):
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    try:
        city_id = int(cb.data[len(CITY_CB_PREFIX):])
    except ValueError:
        await cb.answer()
        return
    try:
        await cb.message.delete()
    except Exception:
        pass
    products = find_products_for_city(city_id)
    if products:
        text5 = get_language_text(lang, "text5") or "Товары, доступные в этом городе"
        await cb.message.answer(text5, reply_markup=build_products_keyboard(lang, products, city_id))
    else:
        text3 = get_language_text(lang, "text3") or "Товары не найдены."
        await cb.message.answer(text3, reply_markup=build_back_keyboard(lang))
    await cb.answer()

@router.callback_query(F.data.startswith(PROD_CB_PREFIX))
async def on_product_clicked(cb: CallbackQuery):
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    payload = cb.data[len(PROD_CB_PREFIX):]
    try:
        pid_str, cid_str = payload.split(":", 1)
        product_id = int(pid_str)
        city_id = int(cid_str)
    except Exception:
        await cb.answer()
        return
    product = get_product(product_id)
    city_name = get_city_name(city_id)
    if not product:
        await cb.answer("Товар не найден", show_alert=True)
        return
    line_city = f"{get_language_text(lang, 'text12') or 'Город'}: {city_name or '-'}"
    line_name = f"{get_language_text(lang, 'text11') or 'Товар'}: {product['product_name'] or '-'}"
    desc = product.get("product_description", "")
    line_desc = desc if desc else ""
    line_tail = f"{get_language_text(lang, 'text6') or ''}".strip()
    parts = [line_city, line_name]
    if line_desc:
        parts.append(line_desc)
    if line_tail:
        parts.append(line_tail)
    caption_text = "\n".join(parts)
    kb = build_product_options_keyboard(lang, product)
    try:
        await cb.message.delete()
    except Exception:
        pass
    raw_path = (product.get("product_image", "") or "").strip()
    is_url, resolved = resolve_image_path(raw_path)
    if resolved:
        try:
            if is_url:
                await cb.message.answer_photo(photo=resolved, caption=caption_text, reply_markup=kb)
            else:
                await cb.message.answer_photo(photo=FSInputFile(resolved), caption=caption_text, reply_markup=kb)
        except Exception as e:
            logger.warning(f"send_photo failed ({resolved}): {e}; fallback to text")
            await cb.message.answer(caption_text, reply_markup=kb)
    else:
        await cb.message.answer(caption_text, reply_markup=kb)
    await cb.answer()

@router.callback_query(F.data.startswith(TAG_CB_PREFIX))
async def on_tag_clicked(cb: CallbackQuery, state: FSMContext):
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    payload = cb.data[len(TAG_CB_PREFIX):]
    try:
        pid_str, idx_str = payload.split(":", 1)
        product_id = int(pid_str)
        tag_index = int(idx_str)
    except Exception:
        await cb.answer()
        return

    product = get_product(product_id)
    if not product:
        await cb.answer("Товар не найден", show_alert=True)
        return

    tags_raw = [
        product.get("product_tags", ""),
        product.get("product_tags2", ""),
        product.get("product_tags3", ""),
        product.get("product_tags4", ""),
    ]
    raw_tag = tags_raw[tag_index] if 0 <= tag_index < len(tags_raw) else ""
    price = parse_tag_price_numeric(raw_tag) or 0.0

    await state.update_data(
        selected_product=product,
        selected_tag={
            'raw': raw_tag,
            'price': price,
            'index': tag_index
        }
    )

    try:
        await cb.message.delete()
    except Exception:
        pass

    text7 = get_language_text(lang, "text7") or "Выберите способ доставки"
    await cb.message.answer(text7, reply_markup=build_delivery_keyboard(lang))
    await cb.answer()

@router.callback_query(F.data.startswith("delivery:"))
async def on_delivery_selected(cb: CallbackQuery, state: FSMContext):
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    delivery_type_raw = cb.data.split(":")[1]
    
    # Преобразуем delivery/pickup в button11/button12
    delivery_type = ""
    if delivery_type_raw == "delivery":
        delivery_type = "button11"  # Доставка
    elif delivery_type_raw == "pickup":
        delivery_type = "button12"  # Самовывоз
    
    state_data = await state.get_data()
    selected_tag = state_data.get('selected_tag')
    price = selected_tag.get('price', 0) if selected_tag else 0
    
    user_balance = get_user_balance(user_id)
    show_balance_button = user_balance >= price if price > 0 else False
    
    # Сохраняем тип доставки в состоянии
    await state.update_data(delivery_type=delivery_type)
    
    print(f"Delivery type saved: {delivery_type_raw} -> {delivery_type}")
    
    try:
        await cb.message.delete()
    except Exception:
        pass

    text10 = get_language_text(lang, "text10") or "Выберите способ оплаты"
    await cb.message.answer(text10, reply_markup=build_payment_keyboard(lang, show_balance_button))
    await cb.answer()

@router.callback_query(F.data.startswith("payment:"))
async def on_payment_selected(cb: CallbackQuery, state: FSMContext):
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    payment_method = cb.data.split(":")[1]
    
    # Логи для отладки передачи типа доставки
    state_data = await state.get_data()
    delivery_type = state_data.get('delivery_type')
    print(f"Payment selected: {payment_method}, Delivery type: {delivery_type}")
    
    if payment_method == "cryptobot":
        await handle_cryptobot_payment(cb, state)
    elif payment_method == "card":
        await handle_bank_card_payment(cb, state)
    elif payment_method == "balance":
        await handle_balance_payment(cb, state)
    
    await cb.answer()

async def handle_balance_payment(cb: CallbackQuery, state: FSMContext):
    """
    Обработчик оплаты с баланса (button16)
    """
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    print(f"=== HANDLE BALANCE PAYMENT ===")
    print(f"User: {user_id}, Lang: {lang}")
    
    # Удаляем сообщение с кнопкой button16
    try:
        await cb.message.delete()
        print("Deleted message with button16")
    except Exception as e:
        print(f"Error deleting message: {e}")
    
    # Получаем данные о товаре из состояния
    state_data = await state.get_data()
    selected_product = state_data.get('selected_product')
    selected_tag = state_data.get('selected_tag')
    
    if not selected_product or not selected_tag:
        error_msg = "Ошибка: данные о товаре не найдены"
        print(error_msg)
        await cb.answer(error_msg, show_alert=True)
        return
    
    price = selected_tag.get('price', 0)
    product_name = selected_product.get('product_name', 'Товар')
    
    print(f"Product: {product_name}, Price: {price}")
    
    if price <= 0:
        error_msg = "Ошибка: неверная цена товара"
        print(error_msg)
        await cb.answer(error_msg, show_alert=True)
        return
    
    # Проверяем баланс пользователя
    user_balance = get_user_balance(user_id)
    
    if user_balance < price:
        error_text = get_language_text(lang, "text18") or "❌ Недостаточно средств на балансе!"
        await cb.message.answer(error_text)
        return
    
    # Списываем средства с баланса
    balance_updated = update_user_balance(user_id, price)
    
    if not balance_updated:
        error_text = "Ошибка списания средств с баланса"
        await cb.message.answer(error_text)
        return
    
    # Сохраняем заказ в базу данных
    from .order_save import save_order
    
    order_saved = await save_order(state_data, user_id, "button16", cb.bot)
    
    if order_saved:
        print("Order successfully saved to database with payment method: button16")
        
        # Показываем text23 после успешной оплаты (БЕЗ КНОПКИ)
        text23 = get_language_text(lang, "text23") or "✅ Оплата прошла успешно с вашего баланса!"
        await cb.message.answer(text23)
        print("Showing text23 after balance payment")
    else:
        print("Failed to save order to database")
        error_text = "Ошибка сохранения заказа"
        await cb.message.answer(error_text)
    
    # Очищаем состояние
    await state.clear()

async def handle_cryptobot_payment(cb: CallbackQuery, state: FSMContext):
    """
    Обработчик оплаты через CryptoBot
    """
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    print(f"=== HANDLE CRYPTOBOT PAYMENT ===")
    print(f"User: {user_id}, Lang: {lang}")
    
    # Получаем данные о товаре из состояния
    state_data = await state.get_data()
    selected_product = state_data.get('selected_product')
    selected_tag = state_data.get('selected_tag')
    delivery_type = state_data.get('delivery_type')
    
    print(f"State data - product: {selected_product is not None}, tag: {selected_tag is not None}")
    print(f"Delivery type: {delivery_type}")
    
    if not selected_product or not selected_tag:
        error_msg = "Ошибка: данные о товаре не найдены"
        print(error_msg)
        await cb.answer(error_msg, show_alert=True)
        return
    
    price = selected_tag.get('price', 0)
    product_name = selected_product.get('product_name', 'Товар')
    
    print(f"Product: {product_name}, Price: {price}")
    
    if price <= 0:
        error_msg = "Ошибка: неверная цена товара"
        print(error_msg)
        await cb.answer(error_msg, show_alert=True)
        return
    
    # Удаляем предыдущее сообщение
    try:
        await cb.message.delete()
        print("Deleted previous message")
    except Exception as e:
        print(f"Error deleting message: {e}")
    
    # Создаем инвойс через функцию из cryptobot.py
    from handlers.cryptobot import create_cryptobot_invoice, build_payment_keyboard
    
    invoice = await create_cryptobot_invoice(
        amount=price,
        description=f"Оплата товара: {product_name}"
    )
    
    if not invoice:
        error_text = "Ошибка: не удалось создать инвойс"
        print(error_text)
        await cb.message.answer(error_text)
        await cb.answer()
        return
        
    if not invoice.get('pay_url'):
        error_text = "Ошибка: не получена ссылка для оплаты"
        print(error_text)
        await cb.message.answer(error_text)
        await cb.answer()
        return
    
    print(f"Invoice created: {invoice['invoice_id']}, URL: {invoice['pay_url']}")
    
    # Сохраняем ID инвойса и сообщения в состоянии для будущей проверки
    await state.update_data(
        cryptobot_invoice_id=invoice['invoice_id'],
        cryptobot_amount=price
    )
    
    # Формируем текст сообщения
    amount_text = get_language_text(lang, "text13") or "Оплатите : {amount} USDT"
    description_text = get_language_text(lang, "text14") or "Для оплаты нажмите кнопку ниже:"
    
    payment_text = f"{amount_text.format(amount=price)}\n\n{description_text}"
    
    # Создаем клавиатуру с кнопками
    keyboard = build_payment_keyboard(lang, invoice['pay_url'])
    
    print(f"Sending payment message with buttons")
    payment_message = await cb.message.answer(payment_text, reply_markup=keyboard)
    
    # Сохраняем ID сообщения с оплатой для удаления при проверке
    await state.update_data(payment_message_id=payment_message.message_id)
    
    await cb.answer()

async def handle_bank_card_payment(cb: CallbackQuery, state: FSMContext):
    """
    Обработчик оплаты банковской картой
    """
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    print(f"=== HANDLE BANK CARD PAYMENT ===")
    print(f"User: {user_id}, Lang: {lang}")
    
    # Получаем данные о товаре из состояния
    state_data = await state.get_data()
    selected_product = state_data.get('selected_product')
    selected_tag = state_data.get('selected_tag')
    
    if not selected_product or not selected_tag:
        error_msg = "Ошибка: данные о товаре не найдены"
        print(error_msg)
        await cb.answer(error_msg, show_alert=True)
        return
    
    price = selected_tag.get('price', 0)
    product_name = selected_product.get('product_name', 'Товар')
    
    print(f"Product: {product_name}, Price: {price}")
    
    # Удаляем предыдущее сообщение
    try:
        await cb.message.delete()
        print("Deleted previous message")
    except Exception as e:
        print(f"Error deleting message: {e}")
    
    # Передаем управление в bank_card.py
    from handlers.bank_card import show_bank_card_payment
    
    await show_bank_card_payment(cb, state, price, product_name)