import logging
from aiogram import Router, F
from aiogram.types import CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
from aiogram.fsm.context import FSMContext
from db import get_user_language, get_language_text, get_button_text

logger = logging.getLogger(__name__)

router = Router(name="bank_card")

def build_card_payment_keyboard(lang: str) -> InlineKeyboardMarkup:
    """
    Создает клавиатуру с кнопкой button18
    """
    button18_text = get_button_text(lang, "button18", "✅ Проверить оплату")
    
    keyboard = InlineKeyboardMarkup(
        inline_keyboard=[
            [InlineKeyboardButton(text=button18_text, callback_data="button18_click")]
        ]
    )
    
    return keyboard

async def show_bank_card_payment(cb: CallbackQuery, state: FSMContext, price: float, product_name: str):
    """
    Показывает информацию об оплате банковской картой
    """
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    # Получаем текст из базы данных
    text18 = get_language_text(lang, "text18") or "Оплата банковской картой"
    
    # Создаем клавиатуру с кнопкой button18
    keyboard = build_card_payment_keyboard(lang)
    
    # Отправляем сообщение с text18 и button18
    await cb.message.answer(text18, reply_markup=keyboard)
    
    print(f"Bank card payment shown for user {user_id}")

@router.callback_query(F.data == "button18_click")
async def handle_button18_click(cb: CallbackQuery, state: FSMContext):
    """
    Обработчик нажатия на button18 - сохраняет заказ и показывает text19
    """
    user_id = cb.from_user.id
    lang = get_user_language(user_id)
    
    print(f"=== BUTTON18 CLICKED ===")
    print(f"User: {user_id}")
    
    # Получаем данные из состояния
    state_data = await state.get_data()
    
    # Сохраняем заказ в базу данных через order_save.py
    from .order_save import save_order
    
    # Сохраняем заказ с payment_method = "button14" (банковская карта)
    order_saved = await save_order(state_data, user_id, "button14", cb.bot)
    
    if order_saved:
        print("Order successfully saved to database with payment method: button14")
        
        # Показываем text19 после успешного сохранения
        text19 = get_language_text(lang, "text19") or "Заказ успешно сохранен"
        await cb.message.answer(text19)
        print("Showing text19 after order save")
    else:
        print("Failed to save order to database")
        error_text = "Ошибка сохранения заказа"
        await cb.message.answer(error_text)
    
    # Очищаем состояние
    await state.clear()
    
    await cb.answer()