import sqlite3
import logging

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

DATA_DB = 'data.db'
MENU_DB = 'bot_menu.db'


def create_db():
    try:
        conn_data = sqlite3.connect(DATA_DB)
        cur = conn_data.cursor()
        
        # Таблица users
        cur.execute('''
            CREATE TABLE IF NOT EXISTS users (
                user_id INTEGER PRIMARY KEY,
                first_name TEXT,
                last_name TEXT,
                username TEXT,
                referal INTEGER DEFAULT 0,
                referal_balance REAL DEFAULT 0,
                referal_link TEXT,
                balance REAL DEFAULT 0,
                orders INTEGER DEFAULT 0,
                user_status BOOLEAN DEFAULT TRUE,
                user_language TEXT
            )
        ''')
        
        # Таблица referals - создаем только если не существует
        cur.execute('''
            CREATE TABLE IF NOT EXISTS referals (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                referals_user_id INTEGER,
                gen_referal_user_id INTEGER,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                UNIQUE(referals_user_id, gen_referal_user_id)
            )
        ''')
        
        conn_data.commit()
        
        # Проверяем и добавляем колонки если нужно
        cur.execute("PRAGMA table_info(users)")
        cols = [r[1] for r in cur.fetchall()]
        if 'user_language' not in cols:
            cur.execute("ALTER TABLE users ADD COLUMN user_language TEXT")
            conn_data.commit()
            
        conn_data.close()
    except Exception as e:
        logger.error(f"Ошибка при создании data.db: {e}")

    try:
        conn_menu = sqlite3.connect(MENU_DB)
        curm = conn_menu.cursor()
        curm.execute('''
            CREATE TABLE IF NOT EXISTS language (
                name TEXT PRIMARY KEY,
                button1 TEXT, button2 TEXT, button3 TEXT, button4 TEXT, button5 TEXT,
                button6 TEXT, button7 TEXT, button8 TEXT, button9 TEXT, button10 TEXT,
                button11 TEXT, button12 TEXT, button13 TEXT, button14 TEXT, button15 TEXT,
                button16 TEXT, button17 TEXT, button18 TEXT, button19 TEXT, button20 TEXT,
                text1 TEXT, text2 TEXT, text3 TEXT, text4 TEXT, text5 TEXT,
                text6 TEXT, text7 TEXT, text8 TEXT, text9 TEXT, text10 TEXT,
                text11 TEXT, text12 TEXT, text13 TEXT, text14 TEXT, text15 TEXT,
                text16 TEXT, text17 TEXT, text18 TEXT, text19 TEXT, text20 TEXT,
                text21 TEXT, text22 TEXT, text23 TEXT, text24 TEXT, text25 TEXT, 
                text26 TEXT, text27 TEXT, text28 TEXT, text29 TEXT, text30 TEXT,
                text31 TEXT
            )
        ''')
        conn_menu.commit()
        conn_menu.close()
    except Exception as e:
        logger.error(f"Ошибка при создании bot_menu.db: {e}")


def add_user(user_id, first_name, last_name, username):
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute('SELECT user_id FROM users WHERE user_id = ?', (user_id,))
        if cur.fetchone():
            conn.close()
            return
        referal_link = f"https://t.me/asdasdass2dasdabot?start={user_id}"
        cur.execute('''
            INSERT INTO users (user_id, first_name, last_name, username, referal_link)
            VALUES (?, ?, ?, ?, ?)
        ''', (user_id, first_name, last_name, username, referal_link))
        conn.commit()
        conn.close()
    except Exception as e:
        logger.error(f"Ошибка при добавлении пользователя {user_id}: {e}")


def get_user_balance(user_id: int) -> float:
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("SELECT balance FROM users WHERE user_id = ?", (user_id,))
        row = cur.fetchone()
        conn.close()
        if row and row[0] is not None:
            try:
                return float(row[0])
            except Exception:
                return 0.0
    except Exception as e:
        logger.error(f"get_user_balance error user_id={user_id}: {e}")
        return 0.0


def update_user_balance(user_id: int, amount: float) -> bool:
    """
    Обновляет баланс пользователя (списание средств)
    """
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("UPDATE users SET balance = balance - ? WHERE user_id = ?", (amount, user_id))
        conn.commit()
        conn.close()
        return True
    except Exception as e:
        logger.error(f"update_user_balance error user_id={user_id}: {e}")
        return False


def add_user_balance(user_id: int, amount: float) -> bool:
    """
    Пополняет баланс пользователя
    """
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("UPDATE users SET balance = balance + ? WHERE user_id = ?", (amount, user_id))
        conn.commit()
        conn.close()
        logger.info(f"Balance added: user {user_id} +{amount}")
        return True
    except Exception as e:
        logger.error(f"add_user_balance error user_id={user_id}: {e}")
        return False


def handle_referal(referals_user_id: int, gen_referal_user_id: int) -> bool:
    """
    Обрабатывает реферальную регистрацию
    Returns: True если реферал успешно обработан, False если уже был обработан
    """
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        
        # Универсальная проверка - используем любую существующую колонку
        cur.execute("PRAGMA table_info(referals)")
        columns = [col[1] for col in cur.fetchall()]
        
        if not columns:
            # Если таблица пустая, создаем базовую запись
            check_query = "SELECT 1 FROM referals WHERE referals_user_id = ? AND gen_referal_user_id = ? LIMIT 1"
        else:
            # Используем первую доступную колонку для проверки
            check_column = columns[0]
            check_query = f"SELECT {check_column} FROM referals WHERE referals_user_id = ? AND gen_referal_user_id = ?"
        
        # Проверяем, не был ли уже этот реферал обработан
        cur.execute(check_query, (referals_user_id, gen_referal_user_id))
        if cur.fetchone():
            conn.close()
            print(f"Referal already processed: {referals_user_id} -> {gen_referal_user_id}")
            return False  # Уже был обработан
        
        # Добавляем запись в таблицу referals
        cur.execute(
            "INSERT INTO referals (referals_user_id, gen_referal_user_id) VALUES (?, ?)",
            (referals_user_id, gen_referal_user_id)
        )
        
        # Увеличиваем счетчик рефералов у пригласившего
        cur.execute(
            "UPDATE users SET referal = referal + 1 WHERE user_id = ?",
            (gen_referal_user_id,)
        )
        
        conn.commit()
        conn.close()
        
        print(f"Referal processed: user {referals_user_id} referred by {gen_referal_user_id}")
        return True
        
    except Exception as e:
        logger.error(f"Ошибка обработки реферала {referals_user_id} -> {gen_referal_user_id}: {e}")
        return False


def get_languages() -> list[str]:
    langs = []
    try:
        conn = sqlite3.connect(MENU_DB)
        cur = conn.cursor()
        cur.execute("SELECT name FROM language ORDER BY name COLLATE NOCASE")
        langs = [r[0] for r in cur.fetchall() if r and r[0]]
        conn.close()
    except Exception as e:
        logger.error(f"Ошибка при чтении языков: {e}")
    return langs


def get_user_language(user_id: int) -> str | None:
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("SELECT user_language FROM users WHERE user_id = ?", (user_id,))
        row = cur.fetchone()
        conn.close()
        return row[0] if row and row[0] else None
    except Exception as e:
        logger.error(f"Ошибка получения языка пользователя {user_id}: {e}")
        return None


def set_user_language(user_id: int, lang: str) -> bool:
    try:
        conn = sqlite3.connect(DATA_DB)
        cur = conn.cursor()
        cur.execute("UPDATE users SET user_language = ? WHERE user_id = ?", (lang, user_id))
        if cur.rowcount == 0:
            cur.execute("INSERT OR IGNORE INTO users (user_id, user_language) VALUES (?, ?)", (user_id, lang))
        conn.commit()
        conn.close()
        return True
    except Exception as e:
        logger.error(f"Ошибка сохранения языка для {user_id}: {e}")
        return False


def get_language_text(lang: str, field: str = "text1") -> str | None:
    if not lang:
        return None
    # Разрешаем все text поля от text1 до text31
    allowed = {f"text{i}" for i in range(1, 32)}
    if field not in allowed:
        field = "text1"
    try:
        conn = sqlite3.connect(MENU_DB)
        cur = conn.cursor()
        cur.execute(f"SELECT {field} FROM language WHERE name = ?", (lang,))
        row = cur.fetchone()
        conn.close()
        return row[0] if row and row[0] else None
    except Exception as e:
        logger.error(f"Ошибка чтения поля {field} для языка {lang}: {e}")
        return None


def get_language_buttons(lang: str) -> dict:
    """
    Возвращает словарь названий кнопок для языка:
    {'button1': '...', 'button2': '...', ..., 'button8': '...'}
    Пустые/NULL значения игнорируются при построении клавиатуры.
    """
    result = {}
    try:
        conn = sqlite3.connect(MENU_DB)
        cur = conn.cursor()
        fields = ",".join([f"button{i}" for i in range(1, 9)])
        cur.execute(f"SELECT {fields} FROM language WHERE name = ?", (lang,))
        row = cur.fetchone()
        conn.close()
        if row:
            for i, val in enumerate(row, start=1):
                if val and str(val).strip():
                    result[f"button{i}"] = str(val).strip()
    except Exception as e:
        logger.error(f"Ошибка чтения кнопок для языка {lang}: {e}")
    return result


def get_button_text(lang: str, column: str, default_value: str = "") -> str:
    """
    Универсальный геттер текста кнопок по столбцу (button11, button12, button16 и т.д.)
    """
    txt = default_value
    try:
        conn = sqlite3.connect(MENU_DB)
        cur = conn.cursor()
        cur.execute(f"SELECT {column} FROM language WHERE name = ?", (lang,))
        row = cur.fetchone()
        conn.close()
        if row and row[0] and str(row[0]).strip():
            txt = str(row[0]).strip()
    except Exception as e:
        logger.error(f"get_button_text error col={column} lang={lang}: {e}")
    return txt