<?php
session_start();
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

$success = $_SESSION['success'] ?? null;
$error = $_SESSION['error'] ?? null;
unset($_SESSION['success'], $_SESSION['error']);

// Fetch cities
$cities = [];
try {
    $pdo = new PDO('sqlite:data.db');
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $stmt = $pdo->query('SELECT * FROM city ORDER BY city_name ASC');
    $cities = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $error = 'Ошибка загрузки городов: ' . $e->getMessage();
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_name = trim($_POST['product_name'] ?? '');
    $product_description = trim($_POST['product_description'] ?? '');
    $selected_cities = isset($_POST['cities']) ? $_POST['cities'] : [];
    $product_city = !empty($selected_cities) ? implode(',', $selected_cities) : '';

    // Tags: 4 pairs of grams and price
    $tags = [];
    $required_tag_filled = false;
    for ($i = 1; $i <= 4; $i++) {
        $grams = trim($_POST["grams$i"] ?? '');
        $price = trim($_POST["price$i"] ?? '');
        if (!empty($grams) && !empty($price)) {
            $tag = $grams . ',' . $price;
            $tags["product_tags$i"] = $tag;
            if ($i === 1) {
                $required_tag_filled = true;
            }
        } else {
            $tags["product_tags$i"] = null;
        }
    }

    // Image upload
    $product_image = null;
    if (isset($_FILES['product_image']) && $_FILES['product_image']['error'] === UPLOAD_ERR_OK) {
        $upload_dir = 'upload/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        $file_extension = pathinfo($_FILES['product_image']['name'], PATHINFO_EXTENSION);
        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif'];
        if (in_array(strtolower($file_extension), $allowed_extensions)) {
            $new_filename = uniqid() . '.' . $file_extension;
            $upload_path = $upload_dir . $new_filename;
            if (move_uploaded_file($_FILES['product_image']['tmp_name'], $upload_path)) {
                $product_image = $upload_path;
            } else {
                $error = 'Ошибка загрузки изображения.';
            }
        } else {
            $error = 'Недопустимый формат изображения. Разрешены: JPG, JPEG, PNG, GIF.';
        }
    }

    // Validation
    if (empty($product_name)) {
        $error = 'Название товара обязательно для заполнения.';
    } elseif (!$required_tag_filled) {
        $error = 'Первый тег (граммы и цена) обязателен для заполнения.';
    } elseif (empty($error)) {
        try {
            $stmt = $pdo->prepare('INSERT INTO products (product_city, product_name, product_image, product_description, product_tags, product_tags2, product_tags3, product_tags4) VALUES (:city, :name, :image, :desc, :tags1, :tags2, :tags3, :tags4)');
            $stmt->execute([
                'city' => $product_city,
                'name' => $product_name,
                'image' => $product_image,
                'desc' => $product_description ?: null,
                'tags1' => $tags['product_tags1'] ?? null,
                'tags2' => $tags['product_tags2'] ?? null,
                'tags3' => $tags['product_tags3'] ?? null,
                'tags4' => $tags['product_tags4'] ?? null
            ]);
            $_SESSION['success'] = 'Товар успешно создан!';
            header('Location: create_product.php');
            exit();
        } catch (PDOException $e) {
            $error = 'Ошибка сохранения товара: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Создать товар - Админ-панель</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-blue-50 to-indigo-100 min-h-screen">
    <?php include 'header.php'; ?>
    
    <!-- Main Content -->
    <main class="max-w-7xl mx-auto py-12 px-4 sm:px-6 lg:px-8">
        <div class="bg-white rounded-2xl shadow-lg p-8">
            <h2 class="text-2xl font-bold text-gray-900 mb-6">Создать товар</h2>
            
            <?php if ($success): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" enctype="multipart/form-data" class="space-y-6">
                <!-- Product Name -->
                <div>
                    <label for="product_name" class="block text-sm font-medium text-gray-700">Название товара</label>
                    <input type="text" id="product_name" name="product_name" required class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500">
                </div>
                
                <!-- Description -->
                <div>
                    <label for="product_description" class="block text-sm font-medium text-gray-700">Описание (необязательно)</label>
                    <textarea id="product_description" name="product_description" rows="3" class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500"></textarea>
                </div>
                
                <!-- Image -->
                <div>
                    <label for="product_image" class="block text-sm font-medium text-gray-700">Изображение товара (необязательно)</label>
                    <input type="file" id="product_image" name="product_image" accept="image/*" class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500">
                </div>
                
                <!-- Cities -->
                <div>
                    <label for="cities" class="block text-sm font-medium text-gray-700">Выберите города (можно несколько)</label>
                    <select id="cities" name="cities[]" multiple class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 h-32">
                        <?php foreach ($cities as $city): ?>
                            <option value="<?php echo htmlspecialchars($city['city_id']); ?>"><?php echo htmlspecialchars($city['city_name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <!-- Tags -->
                <div>
                    <label class="block text-sm font-medium text-gray-700">Теги (варианты грамм и цены)</label>
                    <div class="space-y-4 mt-2">
                        <?php for ($i = 1; $i <= 4; $i++): ?>
                            <div class="flex space-x-4">
                                <div class="flex-1">
                                    <label for="grams<?php echo $i; ?>" class="block text-xs font-medium text-gray-600">Грамм <?php echo $i; ?> (<?php echo ($i === 1 ? 'обязательно' : 'необязательно'); ?>)</label>
                                    <input type="number" id="grams<?php echo $i; ?>" name="grams<?php echo $i; ?>" min="1" step="0.1" class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500" <?php echo ($i === 1 ? 'required' : ''); ?>>
                                </div>
                                <div class="flex-1">
                                    <label for="price<?php echo $i; ?>" class="block text-xs font-medium text-gray-600">Цена <?php echo $i; ?></label>
                                    <input type="number" id="price<?php echo $i; ?>" name="price<?php echo $i; ?>" min="0" step="0.01" class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500" <?php echo ($i === 1 ? 'required' : ''); ?>>
                                </div>
                            </div>
                        <?php endfor; ?>
                    </div>
                </div>
                
                <button type="submit" class="bg-green-600 hover:bg-green-700 text-white px-6 py-2 rounded-lg font-medium transition duration-200">
                    Создать товар
                </button>
            </form>
        </div>
    </main>
</body>
</html>