<?php
session_start();
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

$edit_id = isset($_GET['edit']) ? (int)$_GET['edit'] : null;
$edit_city = null;

// Handle messages from session
$success = $_SESSION['success'] ?? null;
$error = $_SESSION['error'] ?? null;
unset($_SESSION['success'], $_SESSION['error']);

// Handle delete
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete' && isset($_POST['city_id'])) {
    $city_id = (int)$_POST['city_id'];
    try {
        $pdo = new PDO('sqlite:data.db');
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $stmt = $pdo->prepare('DELETE FROM city WHERE city_id = :city_id');
        $stmt->execute(['city_id' => $city_id]);
        $_SESSION['success'] = 'Город успешно удален!';
        header('Location: cities.php');
        exit();
    } catch (PDOException $e) {
        $_SESSION['error'] = 'Ошибка удаления: ' . $e->getMessage();
        header('Location: cities.php');
        exit();
    }
}

// Handle edit/update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update' && isset($_POST['city_id']) && isset($_POST['city_name'])) {
    $city_id = (int)$_POST['city_id'];
    $city_name = trim($_POST['city_name']);
    if (!empty($city_name)) {
        try {
            $pdo = new PDO('sqlite:data.db');
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $stmt = $pdo->prepare('UPDATE city SET city_name = :city_name WHERE city_id = :city_id');
            $stmt->execute(['city_name' => $city_name, 'city_id' => $city_id]);
            $_SESSION['success'] = 'Город успешно обновлен!';
            header('Location: cities.php');
            exit();
        } catch (PDOException $e) {
            $error = 'Ошибка обновления: ' . $e->getMessage();
            $edit_id = $city_id; // Stay in edit mode on error
        }
    } else {
        $error = 'Введите название города!';
        $edit_id = $city_id; // Stay in edit mode on error
    }
}

// Handle add
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add' && isset($_POST['city_name'])) {
    $city_name = trim($_POST['city_name']);
    if (!empty($city_name)) {
        try {
            $pdo = new PDO('sqlite:data.db');
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $stmt = $pdo->prepare('INSERT INTO city (city_name) VALUES (:city_name)');
            $stmt->execute(['city_name' => $city_name]);
            $_SESSION['success'] = 'Город успешно добавлен!';
            header('Location: cities.php');
            exit();
        } catch (PDOException $e) {
            $error = 'Ошибка добавления: ' . $e->getMessage();
        }
    } else {
        $error = 'Введите название города!';
    }
}

// Fetch existing cities
try {
    $pdo = new PDO('sqlite:data.db');
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $stmt = $pdo->query('SELECT * FROM city ORDER BY city_id ASC');
    $cities = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if ($edit_id) {
        $edit_stmt = $pdo->prepare('SELECT * FROM city WHERE city_id = :id');
        $edit_stmt->execute(['id' => $edit_id]);
        $edit_city = $edit_stmt->fetch(PDO::FETCH_ASSOC);
        if (!$edit_city) {
            $edit_id = null; // Invalid ID
        }
    }
} catch (PDOException $e) {
    $cities = [];
    $error = 'Ошибка загрузки городов: ' . $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Города - Админ-панель</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-blue-50 to-indigo-100 min-h-screen">
    <?php include 'header.php'; ?>
    
    <!-- Main Content -->
    <main class="max-w-7xl mx-auto py-12 px-4 sm:px-6 lg:px-8">
        <div class="bg-white rounded-2xl shadow-lg p-8">
            <h2 class="text-2xl font-bold text-gray-900 mb-6">Города</h2>
            
            <?php if ($success): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <!-- Edit Form -->
            <?php if ($edit_id && $edit_city): ?>
                <div class="mb-8 p-6 bg-yellow-50 rounded-lg border border-yellow-200">
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Редактировать город (ID: <?php echo $edit_city['city_id']; ?>)</h3>
                    <form method="POST" class="space-y-4">
                        <input type="hidden" name="action" value="update">
                        <input type="hidden" name="city_id" value="<?php echo $edit_city['city_id']; ?>">
                        <div>
                            <label for="city_name" class="block text-sm font-medium text-gray-700">Название города</label>
                            <input type="text" id="city_name" name="city_name" required class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500" value="<?php echo htmlspecialchars($edit_city['city_name']); ?>">
                        </div>
                        <div class="flex space-x-3">
                            <button type="submit" class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-medium transition duration-200">
                                Сохранить изменения
                            </button>
                            <a href="cities.php" class="bg-gray-500 hover:bg-gray-600 text-white px-6 py-2 rounded-lg font-medium transition duration-200 text-center">
                                Отмена
                            </a>
                        </div>
                    </form>
                </div>
            <?php endif; ?>
            
            <!-- Add City Form -->
            <?php if (!$edit_id): ?>
                <div class="mb-8 p-6 bg-gray-50 rounded-lg">
                    <h3 class="text-lg font-medium text-gray-900 mb-4">Добавить город</h3>
                    <form method="POST" class="space-y-4">
                        <input type="hidden" name="action" value="add">
                        <div>
                            <label for="city_name" class="block text-sm font-medium text-gray-700">Введите название города</label>
                            <input type="text" id="city_name" name="city_name" required class="mt-1 block w-full px-3 py-2 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500" value="<?php echo htmlspecialchars($_POST['city_name'] ?? ''); ?>">
                        </div>
                        <button type="submit" class="bg-green-600 hover:bg-green-700 text-white px-6 py-2 rounded-lg font-medium transition duration-200">
                            Добавить
                        </button>
                    </form>
                </div>
            <?php endif; ?>
            
            <!-- Existing Cities Table -->
            <div class="overflow-x-auto">
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">ID</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Название города</th>
                            <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Действия</th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php if (empty($cities)): ?>
                            <tr>
                                <td colspan="3" class="px-6 py-4 text-center text-gray-500">Нет городов</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($cities as $city): ?>
                                <tr class="hover:bg-gray-50">
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($city['city_id']); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900"><?php echo htmlspecialchars($city['city_name']); ?></td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                                        <a href="?edit=<?php echo $city['city_id']; ?>" class="text-blue-600 hover:text-blue-900">
                                            Редактировать
                                        </a>
                                        <form method="POST" class="inline" onsubmit="return confirm('Удалить город "<?php echo htmlspecialchars($city['city_name']); ?>"?');">
                                            <input type="hidden" name="action" value="delete">
                                            <input type="hidden" name="city_id" value="<?php echo $city['city_id']; ?>">
                                            <button type="submit" class="text-red-600 hover:text-red-900">
                                                Удалить
                                            </button>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </main>
</body>
</html>