<?php
session_start();
if (!isset($_SESSION['logged_in']) || $_SESSION['logged_in'] !== true) {
    header('Location: login.php');
    exit();
}

$config_file = 'handlers/crypto_api.json';
$success = $_SESSION['success'] ?? null;
$error = $_SESSION['error'] ?? null;
unset($_SESSION['success'], $_SESSION['error']);

// Read current config
$current_token = '';
if (file_exists($config_file)) {
    $config_content = file_get_contents($config_file);
    $config = json_decode($config_content, true);
    $current_token = $config['cryptobot_token'] ?? '';
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cryptobot_token'])) {
    $new_token = trim($_POST['cryptobot_token']);
    
    if (!empty($new_token)) {
        try {
            // Prepare new config
            $new_config = [
                'cryptobot_token' => $new_token
            ];
            
            // Ensure directory exists
            if (!is_dir('handlers')) {
                mkdir('handlers', 0755, true);
            }
            
            // Write to file
            $result = file_put_contents($config_file, json_encode($new_config, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
            
            if ($result !== false) {
                $_SESSION['success'] = 'Токен успешно обновлен!';
                header('Location: api.php');
                exit();
            } else {
                $error = 'Ошибка записи в файл. Проверьте права доступа.';
            }
        } catch (Exception $e) {
            $error = 'Ошибка сохранения: ' . $e->getMessage();
        }
    } else {
        $error = 'Введите токен CryptoBot!';
    }
    
    // Update current token for form
    $current_token = $new_token;
}
?>
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>API Настройки - Админ-панель</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-blue-50 to-indigo-100 min-h-screen">
    <?php include 'header.php'; ?>
    
    <!-- Main Content -->
    <main class="max-w-4xl mx-auto py-12 px-4 sm:px-6 lg:px-8">
        <div class="bg-white rounded-2xl shadow-lg p-8">
            <h2 class="text-2xl font-bold text-gray-900 mb-6">Настройки API</h2>
            
            <?php if ($success): ?>
                <div class="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($success); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error): ?>
                <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4">
                    <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <!-- API Token Form -->
            <div class="mb-8 p-6 bg-gray-50 rounded-lg border border-gray-200">
                <h3 class="text-lg font-medium text-gray-900 mb-4">Настройки CryptoBot API</h3>
                
                <form method="POST" class="space-y-6">
                    <div>
                        <label for="cryptobot_token" class="block text-sm font-medium text-gray-700 mb-2">
                            Токен CryptoBot
                        </label>
                        <div class="relative">
                            <input 
                                type="text" 
                                id="cryptobot_token" 
                                name="cryptobot_token" 
                                required 
                                class="mt-1 block w-full px-3 py-3 border border-gray-300 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 text-sm font-mono"
                                value="<?php echo htmlspecialchars($current_token); ?>"
                                placeholder="Введите токен CryptoBot..."
                            >
                            <button 
                                type="button" 
                                onclick="toggleTokenVisibility()" 
                                class="absolute inset-y-0 right-0 pr-3 flex items-center text-gray-400 hover:text-gray-600"
                                title="Показать/скрыть токен"
                            >
                                <svg class="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor" id="eye-icon">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
                                </svg>
                            </button>
                        </div>
                        <p class="mt-2 text-sm text-gray-500">
                            Токен для интеграции с CryptoBot API. Получите его у <a href="https://t.me/CryptoBot" target="_blank" class="text-blue-600 hover:text-blue-900">@CryptoBot</a>
                        </p>
                    </div>
                    
                    <div class="flex items-center justify-between">
                        <div class="text-sm text-gray-500">
                            <?php if (file_exists($config_file)): ?>
                                Файл конфигурации: <code class="bg-gray-100 px-1 rounded">handlers/crypto_api.json</code>
                            <?php else: ?>
                                <span class="text-yellow-600">Файл конфигурации не найден. Будет создан автоматически.</span>
                            <?php endif; ?>
                        </div>
                        <button 
                            type="submit" 
                            class="bg-indigo-600 hover:bg-indigo-700 text-white px-6 py-2 rounded-lg font-medium transition duration-200"
                        >
                            Сохранить токен
                        </button>
                    </div>
                </form>
            </div>
            
            <!-- Current Configuration -->
            <div class="p-6 bg-blue-50 rounded-lg border border-blue-200">
                <h3 class="text-lg font-medium text-blue-900 mb-4">Текущая конфигурация</h3>
                
                <?php if (file_exists($config_file)): ?>
                    <div class="bg-white rounded-lg border border-blue-100 overflow-hidden">
                        <div class="bg-blue-50 px-4 py-2 border-b border-blue-100">
                            <span class="text-sm font-medium text-blue-800">handlers/crypto_api.json</span>
                        </div>
                        <pre class="p-4 text-sm text-gray-800 overflow-x-auto"><code><?php 
                            echo htmlspecialchars(json_encode([
                                'cryptobot_token' => $current_token
                            ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)); 
                        ?></code></pre>
                    </div>
                <?php else: ?>
                    <div class="text-center py-8">
                        <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                        </svg>
                        <h3 class="mt-2 text-sm font-medium text-gray-900">Конфигурация не найдена</h3>
                        <p class="mt-1 text-sm text-gray-500">Файл handlers/crypto_api.json будет создан при сохранении токена.</p>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Instructions -->
            <div class="mt-8 p-6 bg-yellow-50 rounded-lg border border-yellow-200">
                <h3 class="text-lg font-medium text-yellow-900 mb-4">Инструкция по получению токена</h3>
                <ol class="list-decimal list-inside space-y-2 text-sm text-yellow-800">
                    <li>Откройте <a href="https://t.me/CryptoBot" target="_blank" class="text-blue-600 hover:text-blue-900 font-medium">@CryptoBot</a> в Telegram</li>
                    <li>Отправьте команду <code class="bg-yellow-100 px-1 rounded">/start</code></li>
                    <li>Выберите пункт "API" в меню бота</li>
                    <li>Создайте новый API ключ или используйте существующий</li>
                    <li>Скопируйте токен и вставьте в поле выше</li>
                </ol>
            </div>
        </div>
    </main>

    <script>
        function toggleTokenVisibility() {
            const tokenInput = document.getElementById('cryptobot_token');
            const eyeIcon = document.getElementById('eye-icon');
            
            if (tokenInput.type === 'password') {
                tokenInput.type = 'text';
                eyeIcon.innerHTML = '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.88 9.88l-3.29-3.29m7.532 7.532l3.29 3.29M3 3l3.59 3.59m0 0A9.953 9.953 0 0112 5c4.478 0 8.268 2.943 9.543 7a10.025 10.025 0 01-4.132 5.411m0 0L21 21" />';
            } else {
                tokenInput.type = 'password';
                eyeIcon.innerHTML = '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" /><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />';
            }
        }
        
        // Initialize token field as password
        document.addEventListener('DOMContentLoaded', function() {
            const tokenInput = document.getElementById('cryptobot_token');
            tokenInput.type = 'password';
        });
    </script>
</body>
</html>